<?php
require_once __DIR__ . '/../config.php';
setApiHeaders();

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Vérifier l'authentification
requireAuth();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    jsonResponse(['error' => 'Méthode non autorisée'], 405);
}

$input = json_decode(file_get_contents('php://input'), true);
$pdfBase64 = $input['pdf'] ?? '';

if (empty($pdfBase64)) {
    jsonResponse(['error' => 'PDF manquant'], 400);
}

$prompt = 'Analyse ce relevé bancaire marocain et extrais toutes les transactions. Réponds UNIQUEMENT avec un objet JSON valide (sans markdown, sans backticks, sans texte avant ou après) avec cette structure exacte:
{
  "periode": "mois et année du relevé",
  "solde_initial": nombre,
  "solde_final": nombre,
  "total_revenus": nombre,
  "total_depenses": nombre,
  "transactions": [
    {
      "date": "JJ/MM/AAAA",
      "description": "description courte",
      "montant": nombre (négatif pour dépense, positif pour revenu),
      "categorie": "une parmi: Alimentation, Transport, Logement, Loisirs, Santé, Shopping, Abonnements, Restaurants, Transferts, Salaire, Autres"
    }
  ],
  "resume_categories": {
    "Alimentation": nombre total,
    "Transport": nombre total
  },
  "conseils": ["conseil 1", "conseil 2", "conseil 3"],
  "alerte": "message d\'alerte si dépenses importantes détectées ou null"
}';

try {
    switch (AI_PROVIDER) {
        case 'gemini':
            $result = callGemini($pdfBase64, $prompt);
            break;
        case 'openai':
            $result = callOpenAI($pdfBase64, $prompt);
            break;
        case 'claude':
            $result = callClaude($pdfBase64, $prompt);
            break;
        default:
            jsonResponse(['error' => 'Provider IA non configuré'], 500);
    }
    
    jsonResponse(['success' => true, 'analysis' => $result]);
    
} catch (Exception $e) {
    jsonResponse(['error' => $e->getMessage()], 500);
}

// ============================================
// FONCTIONS D'APPEL AUX APIs IA
// ============================================

function callGemini($pdfBase64, $prompt) {
    $apiKey = GEMINI_API_KEY;
    
    if (empty($apiKey) || $apiKey === 'VOTRE_CLE_GEMINI_ICI') {
        throw new Exception('Clé API Gemini non configurée');
    }
    
    $url = "https://generativelanguage.googleapis.com/v1/models/gemini-1.5-flash:generateContent?key=" . $apiKey;
    
    $data = [
        'contents' => [
            [
                'parts' => [
                    [
                        'inline_data' => [
                            'mime_type' => 'application/pdf',
                            'data' => $pdfBase64
                        ]
                    ],
                    ['text' => $prompt]
                ]
            ]
        ]
    ];
    
    $response = makeRequest($url, $data);
    
    if (isset($response['error'])) {
        throw new Exception($response['error']['message'] ?? 'Erreur Gemini API');
    }
    
    $text = $response['candidates'][0]['content']['parts'][0]['text'] ?? '';
    return parseAIResponse($text);
}

function callOpenAI($pdfBase64, $prompt) {
    $apiKey = OPENAI_API_KEY;
    
    if (empty($apiKey)) {
        throw new Exception('Clé API OpenAI non configurée');
    }
    
    // Note: OpenAI ne supporte pas les PDFs directement
    // Il faudrait convertir en image d'abord
    throw new Exception('OpenAI ne supporte pas les PDFs. Utilisez Gemini.');
}

function callClaude($pdfBase64, $prompt) {
    $apiKey = CLAUDE_API_KEY;
    
    if (empty($apiKey)) {
        throw new Exception('Clé API Claude non configurée');
    }
    
    $url = "https://api.anthropic.com/v1/messages";
    
    $data = [
        'model' => 'claude-sonnet-4-20250514',
        'max_tokens' => 4000,
        'messages' => [
            [
                'role' => 'user',
                'content' => [
                    [
                        'type' => 'document',
                        'source' => [
                            'type' => 'base64',
                            'media_type' => 'application/pdf',
                            'data' => $pdfBase64
                        ]
                    ],
                    ['type' => 'text', 'text' => $prompt]
                ]
            ]
        ]
    ];
    
    $headers = [
        'Content-Type: application/json',
        'x-api-key: ' . $apiKey,
        'anthropic-version: 2023-06-01'
    ];
    
    $response = makeRequest($url, $data, $headers);
    
    if (isset($response['error'])) {
        throw new Exception($response['error']['message'] ?? 'Erreur Claude API');
    }
    
    $text = '';
    foreach ($response['content'] ?? [] as $item) {
        if ($item['type'] === 'text') {
            $text .= $item['text'];
        }
    }
    
    return parseAIResponse($text);
}

function makeRequest($url, $data, $customHeaders = null) {
    $headers = $customHeaders ?? ['Content-Type: application/json'];
    
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($data),
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 120
    ]);
    
    $response = curl_exec($ch);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        throw new Exception('Erreur de connexion: ' . $error);
    }
    
    return json_decode($response, true);
}

function parseAIResponse($text) {
    // Nettoyer le JSON
    $text = preg_replace('/```json\s*/', '', $text);
    $text = preg_replace('/```\s*/', '', $text);
    $text = trim($text);
    
    $result = json_decode($text, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Erreur de parsing de la réponse IA');
    }
    
    return $result;
}
